const jwt = require("jsonwebtoken");
const { hashPassword, comparePassword } = require("../utils/auth");


exports.getAdmins = async (req, res) => {
    const db = req.db
    const selectQuery = 'SELECT * FROM admins';

    db.query(selectQuery, (err, results) => {

        if (results?.length === 0) {
            return res.status(404).json({ message: "No user found" })
        }

        if (err) {
            console.error('Error retrieving clients:', err);
            return res.status(500).json({ error: 'Internal Server Error' });
        }

        res.status(200).json(results);
    })

};

exports.getSingleAdmin = async (req, res) => {
    const db = req.db;
    const adminId = req.params.id;
    const selectQuery = 'SELECT * FROM admins WHERE id = ?'
    const value = [adminId]

    db.query(selectQuery, value, (err, results) => {
        if (err) {
            console.error('error retriving:', err)
            return res.status(500).json({ error: 'Internal Server Error' })
        }

        if (results.length === 0) {
            return res.status(404).json({ message: 'admin not found.' });
        }

        const admin = results[0];
        res.status(200).json(admin);

    })

};

exports.login = async (req, res) => {
    const { email, password } = req.body;

    if (!email || !password) {
        return res.status(400).json({ error: "Email and password are required" });
    }

    const db = req.db;
    const selectQuery = 'SELECT id, name, email, phone, password, role FROM admins WHERE email = ?'; // Select only needed fields

    db.query(selectQuery, [email], async (err, results) => {
        if (err) {
            console.error('Error during login:', err);
            return res.status(500).json({ error: 'Internal Server Error' });
        }

        if (results.length === 0) {
            return res.status(401).json({ error: 'Invalid credentials' });
        }

        const admin = results[0];
        const passwordMatch = await comparePassword(password, admin.password); // Compare with hashed password

        if (!passwordMatch) {
            return res.status(401).json({ error: 'Invalid credentials' });
        }

        // Generate JWT token
        const token = jwt.sign({ id: admin.id, role: admin.role }, process.env.JWT_SECRET, { expiresIn: '1h' });

        // Create user object with selected data
        const user = {
            id: admin.id,
            name: admin.name,
            email: admin.email,
            phone: admin.phone,
            role: admin.role
        };

        res.status(200).json({ token, user });
    });
};

exports.logout = async (req, res) => {
    const authHeader = req.headers['authorization'];
    const token = authHeader && authHeader.split(' ')[1];

    if (token) {
        tokenBlacklist.add(token);
    }
    res.status(200).json({ message: 'Logged out successfully' });
};

exports.createAdmin = async (req, res) => {
    const { name, email, phone, password, role } = req.body;

    if (!email || !password) {
        return res.status(400).json({ error: 'Email, and password are required' });
    }

    const db = req.db;
    const hashedPassword = await hashPassword(password);

    const insertQuery = 'INSERT INTO admins (name, email, phone, password, role) VALUES (?, ?, ?, ?, ?)';
    const values = [name, email, phone, hashedPassword, role];

    db.query(insertQuery, values, (err, result) => {
        if (err) {
            console.error('Error creating admin:', err);
            return res.status(500).json({ error: 'Internal Server Error' });
        }
        res.status(201).json({ message: 'Admin created successfully', adminId: result.insertId });
    });
};

exports.updateProfile = async (req, res) => {
    // 1. Authentication Check (Assumes middleware for token verification)
    if (!req.user) { // Check if req.user is set by the authentication middleware
        return res.status(401).json({ error: 'Unauthorized' });
    }

    const adminId = req.user.id; // Get ID from authenticated user
    const { name, email, phone } = req.body;

    // 2. Input Validation
    if (!email) {
        return res.status(400).json({ error: 'Email required' });
    }

    const db = req.db;

    // 3. Update Query (Using parameterized query for security)
    const updateQuery = 'UPDATE admins SET name = ?, email = ?, phone = ? WHERE id = ?';
    const values = [name, email, phone, adminId];

    db.query(updateQuery, values, (err, result) => {
        if (err) {
            console.error('Error updating admin profile:', err);
            return res.status(500).json({ error: 'Internal Server Error' });
        }
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Admin not found' });
        }
        res.status(200).json({ message: 'Profile updated successfully' });
    });
};

exports.deleteAdminById = async (req, res) => {
    // 1. Authentication Check (Assumes middleware for token verification)
    if (!req.user) {
        return res.status(401).json({ error: 'Unauthorized' });
    }

    // 2. Authorization Check (Only admins with 'superAdmin' role can delete)
    if (req.user.role !== 'SuperAdmin') {
        return res.status(403).json({ error: 'Forbidden: Only superAdmins can delete other admins' });
    }

    const adminId = req.params.id;
    const db = req.db;

    // 3. Check if this is the last superAdmin
    const countQuery = 'SELECT COUNT(*) as count FROM admins WHERE role = "SuperAdmin"';

    db.query(countQuery, (err, results) => {
        if (err) {
            console.error('Error counting superAdmins:', err);
            return res.status(500).json({ error: 'Internal Server Error' });
        }

        const superAdminCount = results[0].count;

        if (superAdminCount === 1 && req.params.id === req.user.id) { // Check if deleting self and only one superAdmin
            return res.status(400).json({ error: 'Cannot delete the last superAdmin' });
        }

        // 4. Delete Query (Using parameterized query for security)
        const deleteQuery = 'DELETE FROM admins WHERE id = ?';
        db.query(deleteQuery, [adminId], (err, result) => {
            if (err) {
                console.error('Error deleting admin:', err);
                return res.status(500).json({ error: 'Internal Server Error' });
            }
            if (result.affectedRows === 0) {
                return res.status(404).json({ error: 'Admin not found' });
            }
            res.status(200).json({ message: 'Admin deleted successfully' });
        });
    });
};

